//	Maze4DContentView.swift
//
//	© 2025 by Jeff Weeks
//	See TermsOfUse.txt

import SwiftUI


enum PanelType {
	case noPanel
	case newMazePanel
	case optionsPanel
	case shearPanel
	case exportPanel
	case helpMenuPanel
	case helpPanel
	case contactPanel
	case translatorsPanel
}

struct Maze4DContentView: View {

	@State var itsModelData = Maze4DModel()

	@State var itsActivePanel = PanelType.newMazePanel

	@AppStorage("Inertia") var itsInertia: Bool = true

	//	Let's give our control panels a background color which,
	//	in dark mode, isn't quite black, to provide some contrast
	//	against the main view's pure black background.
	//
	//		Note: itsModifiedBackgroundColor will change on the fly,
	//		if the player switches between light mode and dark mode.
	//
	@State var itsModifiedBackgroundColor = Color("Modified Background Color")

	//	To make App Store screenshots, set gMakeScreenshots = true.
	@State var itsScreenshotIndex: Int = gFirstScreenshotIndex

	@State var itsRenderer = Maze4DRenderer()

	@GestureState var itsDragState: Maze4DDragState = Maze4DDragState()
	@GestureState var itsPreviousRotationAngle: Double = 0.0	//	in radians


	var body: some View {
	
		let theTranslatorThanks = [
			GeometryGamesTranslatorThanks(pre: "mit Dank an", name: "Bernd Macheleidt und\nSascha Gritzbach", post: nil, lang: "de"),
			GeometryGamesTranslatorThanks(pre: "grazie a", name: "Daniela Bigatti", post: nil, lang: "it"),
			GeometryGamesTranslatorThanks(pre: nil, name: "竹内建氏に", post: "感謝します", lang: "ja"),
			GeometryGamesTranslatorThanks(pre: "z podziękowaniami dla", name: "Marcina Pilarskiego", post: nil, lang: "pl"),
			GeometryGamesTranslatorThanks(pre: "obrigado a", name: "Nuno Fernandes", post: nil, lang: "pt"),
			GeometryGamesTranslatorThanks(pre: "спасибо", name: "Маше Трнковой", post: nil, lang: "ru"),
			GeometryGamesTranslatorThanks(pre: nil, name: "Süleyman Saltuk Güngör'e", post: "teşekkürler", lang: "tr")
		]

#if os(iOS)
		let theEdgesToExpandFromSafeArea: Edge.Set = .all
#endif
#if os(macOS)
		let theEdgesToExpandFromSafeArea: Edge.Set = .bottom
#endif

		ZStack(alignment: .bottom) {
		
			//	Main animation view
			if let theRenderer = itsRenderer {

				GeometryReader() { geometryReaderProxy in
				
					GeometryGamesViewRep(
						modelData: itsModelData,
						renderer: theRenderer
					)
					//
					//	Gesture order matters:
					//		SwiftUI considers each Gesture only after
					//		all preceding gestures have failed.
					//
					.gesture(
						maze4DDragGesture(
							modelData: itsModelData,
							viewSize: geometryReaderProxy.size,
							inertia: itsInertia,
							dragState: $itsDragState))
					.gesture(
						maze4DRotateGesture(
							modelData: itsModelData,
							previousAngle: $itsPreviousRotationAngle))
					.gesture(
						TapGesture()
						.onEnded() { _ in
							if itsActivePanel != .noPanel {
								itsActivePanel = .noPanel
							} else {
								itsModelData.itsIncrement = nil
							}
						}
					)
				}
				.ignoresSafeArea(edges: theEdgesToExpandFromSafeArea)
			}
			
			VStack() {
			
				//	The active panel, if any
				Group() {
				
					//	Note:  Some of the panels get leading-aligned, others trailing-aligned.
					switch itsActivePanel {

					case .noPanel:
						Spacer()

					case .newMazePanel:
						if let theRenderer = itsRenderer {
							NewMazeView(
								modelData: itsModelData,
								activePanel: $itsActivePanel,
								renderer: theRenderer)
							.frame(maxWidth: .infinity, alignment: .leading)
						}
					
					case .optionsPanel:
						OptionsView(inertia: $itsInertia)
						.frame(maxWidth: .infinity, alignment: .leading)
						.onChange(of: itsInertia) {
							itsModelData.itsIncrement = nil
						}

					case .shearPanel:
						if let theRenderer = itsRenderer {
							ShearView(
								modelData: itsModelData,
								renderer: theRenderer)
							.frame(maxWidth: .infinity, alignment: .trailing)
						}

					case .exportPanel:
						if let theRenderer = itsRenderer {
							GeometryGamesExportView(
								modelData: itsModelData,
								exportRenderer: theRenderer,
								caption: "4D Maze image",
								modifiedBackgroundColor: itsModifiedBackgroundColor)
							.frame(maxWidth: .infinity, alignment: .trailing)
						} else {
							Text("Renderer not available")
						}
					
					case .helpMenuPanel:
						HelpMenuView(activePanel: $itsActivePanel)
						.frame(maxWidth: .infinity, alignment: .trailing)
								
					case .helpPanel:
						Maze4DHelpView()
						.frame(maxWidth: .infinity, alignment: .trailing)

					case .contactPanel:
						GeometryGamesContactView(
							modifiedBackgroundColor: itsModifiedBackgroundColor)
						.frame(maxWidth: .infinity, alignment: .trailing)

					case .translatorsPanel:
						GeometryGamesTranslatorsView(
							theTranslatorThanks,
							modifiedBackgroundColor: itsModifiedBackgroundColor)
						.frame(maxWidth: .infinity, alignment: .trailing)
					}
				}
			
				//	The buttons
				HStack(alignment: .bottom) {
					NewMazeButton(activePanel: $itsActivePanel)
					OptionsButton(activePanel: $itsActivePanel)
					Spacer()
					ShearButton(activePanel: $itsActivePanel)
					ExportButton(activePanel: $itsActivePanel)
					HelpMenuButton(activePanel: $itsActivePanel)
				}
				.modifier(applyForegroundAccentColorOnMacOS())
			}
			.padding(geometryGamesControlInset)
				
			if gMakeScreenshots && !gGetScreenshotOrientations {
				//	The Rectangle will respond to taps
				//	only if it's at least a tiny bit opaque.
				Rectangle()
				.foregroundStyle(Color(.displayP3,
									red: 0.0,
									green: 0.0,
									blue: 0.0,
									opacity: 0.000000000001))
				.ignoresSafeArea()
				.onTapGesture {
#if os(iOS)
					switch itsScreenshotIndex {
					case 0: itsScreenshotIndex = 1	//	toggle between language-independent screenshots
					case 1: itsScreenshotIndex = 0
					case 2: itsScreenshotIndex = 3	//	toggle between language-dependent screenshots
					case 3: itsScreenshotIndex = 2
					default: assertionFailure("Unexpected value of itsScreenshotIndex")
					}
#endif
#if os(macOS)
					itsScreenshotIndex = (itsScreenshotIndex + 1) % 4
#endif
				}
			}
		}
		.modifier(buttonStyleOnMacOS())
		.onAppear() {
			if gMakeScreenshots {
				prepareForAppStoreScreenshot(
					itsScreenshotIndex,
					modelData: itsModelData,
					activePanel: $itsActivePanel,
					optionalRenderer: itsRenderer)
			}
		}
		.onChange(of: itsScreenshotIndex) {
			prepareForAppStoreScreenshot(
				itsScreenshotIndex,
				modelData: itsModelData,
				activePanel: $itsActivePanel,
				optionalRenderer: itsRenderer)
		}
	}
}


struct NewMazeButton: View {

	@Binding var activePanel: PanelType

	var body: some View {

		Button {
			activePanel = activePanel == .newMazePanel ? .noPanel : .newMazePanel
		} label: {
			Image(systemName: "arrow.clockwise")
				.font(.title)
				.padding(geometryGamesControlPadding)
		}
	}
}

struct NewMazeView: View {

	var modelData: Maze4DModel
	@Binding var activePanel: PanelType
	let renderer: Maze4DRenderer

	@State var modifiedBackgroundColor = Color("Modified Background Color")

	var body: some View {

		VStack(alignment: .leading) {

			ForEach(DifficultyLevel.allCases, id: \.id) { difficultyLevel in

				Button(
					action: {
						modelData.itsMaze = makeNewMaze(difficultyLevel)
						renderer.refreshForNewMaze(modelData: modelData)
						activePanel = .noPanel
					},
					label: {

						let n = difficultyLevel.n
								
						HStack(spacing: 16.0) {

							Circle()
							.fill(difficultyLevel.color)
							.frame(width: 32.0, height: 32.0)

							VStack(alignment: .leading) {
							
								Text(difficultyLevel.localizedNameKey)
								.lineLimit(1)
								.modifier(applyForegroundAccentColorOnMacOS())

								Text(verbatim: "\(n) × \(n) × \(n) × \(n)")
								.foregroundStyle(Color.gray)
								.font(.footnote)
							}
						}
					}
				)
			}
		}
		.fixedSize()
		.padding(geometryGamesPanelPadding)
		.background(modifiedBackgroundColor)
		.cornerRadius(geometryGamesCornerRadius)
	}
}


struct OptionsButton: View {

	@Binding var activePanel: PanelType

	var body: some View {

		Button {
			activePanel = activePanel == .optionsPanel ? .noPanel : .optionsPanel
		} label: {
			Image(systemName: "switch.2")
				.font(.title)
				.padding(geometryGamesControlPadding)
		}
	}
}

struct OptionsView: View {

	@Binding var inertia: Bool

	@State var modifiedBackgroundColor = Color("Modified Background Color")

	var body: some View {
	
		Toggle("Inertia", isOn: $inertia)
		.fixedSize()
		.padding(geometryGamesPanelPadding)
		.background(modifiedBackgroundColor)
		.cornerRadius(geometryGamesCornerRadius)
	}
}


struct ShearButton: View {

	@Binding var activePanel: PanelType

	var body: some View {

		Button {
			activePanel = activePanel == .shearPanel ? .noPanel : .shearPanel
		} label: {
			Image(systemName: "arrow.left.arrow.right.circle")
				.font(.title)
				.padding(geometryGamesControlPadding)
		}
	}
}

struct ShearView: View {

	@Bindable var modelData: Maze4DModel
	let renderer: Maze4DRenderer

	@State var modifiedBackgroundColor = Color("Modified Background Color")

	var body: some View {

		VStack() {
			Text("4D Shear")
			Slider(
				value: $modelData.its4DShearFactor,
				in: min4DShearFactor ... max4DShearFactor
			)
			.onChange(of: modelData.its4DShearFactor) {
				renderer.refreshForNewShear(modelData: modelData)
			}
		}
		.padding(geometryGamesPanelPadding)
		.background(modifiedBackgroundColor)
		.cornerRadius(geometryGamesCornerRadius)
		.frame(width: 256.0)
	}
}


struct ExportButton: View {

	@Binding var activePanel: PanelType

	var body: some View {

		Button {
			activePanel = activePanel == .exportPanel ? .noPanel : .exportPanel
		} label: {
			Image(systemName: "square.and.arrow.up")
				.font(.title)
				.padding(geometryGamesControlPadding)
		}
	}
}


struct HelpMenuButton: View {

	@Binding var activePanel: PanelType

	var body: some View {

		Button {
			activePanel = activePanel == .helpMenuPanel ? .noPanel : .helpMenuPanel
		} label: {
			Image(systemName: "questionmark.circle")
				.font(.title)
				.padding(geometryGamesControlPadding)
				.background(geometryGamesTappableClearColor)
		}
	}
}

struct HelpMenuView: View {

	@Binding var activePanel: PanelType

	var body: some View {

		VStack(alignment: .leading, spacing: 8.0) {

			Button() {
				activePanel = .helpPanel
			} label: {
				Label("4D Maze Help", systemImage: "questionmark.circle")
			}

			Button() {
				activePanel = .contactPanel
			} label: {
				GeometryGamesHelpMenuContactItemLabel()
			}

			Button() {
				activePanel = .translatorsPanel
			} label: {
				GeometryGamesHelpMenuTranslatorsItemLabel()
			}
		}
		.modifier(helpMenuStyle())
	}
}

let headerColor = Color(.displayP3, red: 0.0, green: 0.5, blue: 0.25)

//	Avoid italics in non-European languages.
let gFontDesign = {
	let theLanguage = Locale.preferredLanguages.first ?? "en"
	switch theLanguage {
		case "ja": return Font.Design.default
		default:   return Font.Design.serif
	}
}()

struct Maze4DHelpView: View {

	@State var modifiedBackgroundColor = Color("Modified Background Color")

	var body: some View {

		ScrollView {
			VStack(alignment: .leading, spacing: 16) {

				Text("4D Maze")
					.font(.title2)
					.foregroundStyle(headerColor)
					.padding(EdgeInsets(top: 16.0, leading: 0.0, bottom: 0.0, trailing: 0.0))
					.frame(maxWidth: .infinity, alignment: .center)

				Maze4DHelpSection(title: "HowToPlay-title") {
				
					Text("HowToPlay-text")
						.modifier(Maze4DBodyText())
					Image("Help/SliderToBox")
						.resizable()
						.aspectRatio(contentMode: .fit)
						.frame(width: 192.0)
				}

				Maze4DHelpSection(title: "AnaKata-title") {
				
					Text("AnaKata-text")
						.modifier(Maze4DBodyText())
					Spacer()
						.frame(height: 4.0)
					HStack(spacing: 0.0) {
						VStack(alignment: .trailing) {
							Text("AnaKata-label1")
							Text("AnaKata-label2")
							Text("AnaKata-label3")
							Text("AnaKata-label4")
						}
						Spacer()
							.frame(width: 8.0)
						VStack(alignment: .trailing) {
							Text("AnaKata-pos1")
							Text("AnaKata-pos2")
							Text("AnaKata-pos3")
							Text("AnaKata-pos4")
						}
						Spacer()
							.frame(width: 2.0)
						VStack() {
							Text(verbatim: "/")
							Text(verbatim: "/")
							Text(verbatim: "/")
							Text(verbatim: "/")
						}
						Spacer()
							.frame(width: 2.0)
						VStack(alignment: .leading) {
							Text("AnaKata-neg1")
							Text("AnaKata-neg2")
							Text("AnaKata-neg3")
							Text("AnaKata-neg4")
						}
					}
					.font(.system(size: 15.0, design: gFontDesign))
				}

				Maze4DHelpSection(title: "Colors-title") {
				
					Text("Colors-text1")
						.modifier(Maze4DBodyText())
					Text("Colors-caption1")
						.modifier(Maze4DCaptionText())
					Image("Help/ColoredTubes")
						.resizable()
						.aspectRatio(contentMode: .fit)
						.frame(width: 256.0)
						
					Spacer()
						.frame(height: 16.0)
						
					Text("Colors-text2")
						.modifier(Maze4DBodyText())
					Text("Colors-caption2")
						.modifier(Maze4DCaptionText())
					Image("Help/RainbowTube")
						.resizable()
						.aspectRatio(contentMode: .fit)
						.frame(width: 56.0)
				}

				Maze4DHelpSection(title: "SuperimposedPoints-title") {
				
					Text("SuperimposedPoints-text")
						.modifier(Maze4DBodyText())
				}

				Maze4DHelpSection(title: "4DConnections-title") {
				
					Text("4DConnections-text")
						.modifier(Maze4DBodyText())

					//	Even a tiny bit of top padding on the HStack
					//	seems to add a whole line of empty space.
					//	So let's use a Spacer instead.
					Spacer()
						.frame(height: 8.0)

					//	Side-by-side captions
					HStack(alignment: .center, spacing: 8.0) {
						Text("4DConnections-caption1")
							.multilineTextAlignment(.center)
							.font(.caption)
							.frame(width: 128.0)
						Text("4DConnections-caption2")
							.multilineTextAlignment(.center)
							.font(.caption)
							.frame(width: 128.0)
					}

					//	Side-by-side figures
					HStack(alignment: .center, spacing: 8.0) {
						Image("Help/ConnectedTubes")
							.resizable()
							.aspectRatio(contentMode: .fit)
							.frame(width: 128.0)
						Image("Help/UnconnectedTubes")
							.resizable()
							.aspectRatio(contentMode: .fit)
							.frame(width: 128.0)
					}
				}
				
				Text("Questions-footnote")
					.font(.system(.footnote, design: gFontDesign))
				
				Text(verbatim: "© 2025 by Jeff Weeks")
					.font(.system(.footnote, design: gFontDesign))
			}
		}
		.padding(geometryGamesPanelPadding)
		.background(modifiedBackgroundColor)
		.cornerRadius(geometryGamesCornerRadius)
		.frame(width: 352.0)
		.frame(maxHeight: .infinity)
	}
}

struct Maze4DHelpSection<Content: View>: View {
	
	let title: LocalizedStringKey
	let content: Content
	
	init(
		title: LocalizedStringKey,
		@ViewBuilder content: () -> Content
	) {
	
		self.title = title
		self.content = content()
	}

	var body: some View {

		VStack(alignment: .center) {

			Text(title)
				.font(.title3)
				.foregroundStyle(headerColor)
				.frame(maxWidth: .infinity, alignment: .leading)

			Spacer()
				.frame(height: 6.0)
			
			content
		}
	}
}

struct Maze4DBodyText: ViewModifier {

	func body(content: Content) -> some View {

		content
			.font(.system(.body, design: .serif))
			.padding(EdgeInsets(top: 0.0, leading: 16.0, bottom: 0.0, trailing: 0.0))
			.frame(maxWidth: .infinity, alignment: .leading)
	}
}

struct Maze4DCaptionText: ViewModifier {

	func body(content: Content) -> some View {

		//	Even a tiny bit of top padding on the content
		//	seems to add a whole line of empty space.
		//	So let's set the top padding to zero
		//	and use a Spacer instead.
		Spacer()
			.frame(height: 8.0)
			
		content
			.multilineTextAlignment(.center)
			.font(.caption)
			.padding(EdgeInsets(top: 0.0, leading: 20.0, bottom: 0.0, trailing: 20.0))
	}
}
